function Get-AntiMalwareStatus {
 
# .SYNOPSIS
# Get-AnitMalewareStatus is an advanced Powershell function.
# It gets the Windows Defender Status of the local computer and remote computer.
 
# .DESCRIPTION
# Uses Invoke-Command and Get-MpComputerStatus.
 
# .PARAMETER
# Scope
# Define a scope. Possible values:
# AllServer, AllComputer
# Default: localhost
 
# .EXAMPLE
# Get-AntiMalwareStatus -Scope AllComputer
 
# .NOTES
# Author: Patrick Gruenauer
# Web: https://sid-500.com 
 
[CmdletBinding()]
 
param
 
(
 
[Parameter(Position=0,Helpmessage = 'Possible Values: AllServer, AllComputer')]
[ValidateSet('AllServer','AllComputer')]
$Scope
 
)
 
$result=@()
$ErrorActionPreference="SilentlyContinue"
switch ($Scope) {
 
$null {
 
Get-MpComputerStatus | Select-Object -Property Antivirusenabled,AMServiceEnabled,AntispywareEnabled,BehaviorMonitorEnabled,IoavProtectionEnabled,`
NISEnabled,OnAccessProtectionEnabled,RealTimeProtectionEnabled,AntivirusSignatureLastUpdated
 
}
 
AllServer {
 
$server=Get-ADComputer -Filter 'operatingsystem -like "*server*" -and enabled -eq "true"' | Select-Object -ExpandProperty Name
 
foreach ($s in $server) {
 
$rs=Invoke-Command -ComputerName $s {Get-MpComputerStatus | Select-Object -Property Antivirusenabled,AMServiceEnabled,AntispywareEnabled,` BehaviorMonitorEnabled,IoavProtectionEnabled,NISEnabled,OnAccessProtectionEnabled,RealTimeProtectionEnabled,AntivirusSignatureLastUpdated}
 
If ($rs) {
 
$result+=New-Object -TypeName PSObject -Property ([ordered]@{
 
'Server'=$rs.PSComputername
'Anti-Virus'=$rs.AntivirusEnabled
'AV Update'=$rs.AntivirusSignatureLastUpdated
'Anti-Malware'=$rs.AMServiceEnabled
'Anti-Spyware'=$rs.AntispywareEnabled
'Behavior Monitor'=$rs.BehaviorMonitorEnabled
'Office-Anti-Virus'=$rs.IoavProtectionEnabled
'NIS'=$rs.NISEnabled
'Access Prot'=$rs.OnAccessProtectionEnabled
'R-T Prot'=$rs.RealTimeProtectionEnabled
 
})
 
}
 
}
}
 
AllComputer {
 
$comp=Get-ADComputer -Filter 'enabled -eq "true"' | Select-Object -ExpandProperty Name
 
foreach ($c in $comp) {
 
$rs=Invoke-Command -ComputerName $c {Get-MpComputerStatus | Select-Object -Property Antivirusenabled,AMServiceEnabled,AntispywareEnabled,` BehaviorMonitorEnabled,IoavProtectionEnabled,NISEnabled,OnAccessProtectionEnabled,RealTimeProtectionEnabled,AntivirusSignatureLastUpdated}
 
If ($rs) {
 
$result+=New-Object -TypeName PSObject -Property ([ordered]@{
 
'Computer'=$rs.PSComputername
'Anti-Virus'=$rs.AntivirusEnabled
'AV Update'=$rs.AntivirusSignatureLastUpdated
'Anti-Malware'=$rs.AMServiceEnabled
'Anti-Spyware'=$rs.AntispywareEnabled
'Behavior Monitor'=$rs.BehaviorMonitorEnabled
'Office-Anti-Virus'=$rs.IoavProtectionEnabled
'NIS'=$rs.NISEnabled
'Access Prot'=$rs.OnAccessProtectionEnabled
'R-T Prot'=$rs.RealTimeProtectionEnabled
 
})
 
}
}
}
 
}
Write-Output $result
}

Get-AntiMalwareStatus | Out-File -FilePath .\1-Windows_Defender_status.txt